% Copyright (c) 2016 Giampaolo D'Alessandro
% Mathematical Sciences - University of Southampton
% Email: dales@soton.ac.uk
% 
% Permission is hereby granted, free of charge, to any person obtaining a copy
% of this software and associated documentation files (the "Software"), to deal
% in the Software without restriction, including without limitation the rights
% to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
% copies of the Software, and to permit persons to whom the Software is
% furnished to do so, subject to the following conditions:
% 
% The above copyright notice and this permission notice shall be included in
% all copies or substantial portions of the Software.
% 
% THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
% IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
% FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
% AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
% LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
% OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
% SOFTWARE.

function [theta] = DirectorProfile(V,delta_1,delta_2,alpha_psi,...
  theta_L,theta_R,z)
% Compute the director profile for a range of voltages
%
% The director profile is the solution of the (non-dimensional) equation
% %
% \begin{equation*}
%    \frac{\partial \theta}{\partial t} = 
%    [1-\delta_2 \cos(2 \theta)] \theta'' +
%    \delta_{2} \sin(2 \theta) \theta'^{2} +
%    \delta_{1} \sin(2 \theta) \psi^2 
% \end{equation*}
% %
% where
% %
% \begin{equation*}
%   \delta_{2} = \frac{K_{3}-K_{1}}{K_{3}+K_{1}}, \quad
%   \delta_{1} = \epsilon_{\|} - \epsilon_{\perp} ,
% \end{equation*}
% %
% and $\psi(z)$ is the electric field, given by
% %
% \begin{equation*}
%    \psi = \frac{V}{1-\alpha_{\psi} \cos(2 \theta)}
%    \left [
%      \int_{0}^{1} \frac{d z'}{1-\alpha_{\psi} \cos(2 \theta)}
%    \right ]^{-1}
% \end{equation*}
% %
% with
% %
% \begin{equation*}
%    alpha_{\psi} = \frac{\epsilon_{\|} - \epsilon_{\perp}}
%                 {\epsilon_{\|} + \epsilon_{\perp}} \, .
% \end{equation*}

% % Compute the average temperature assuming that the absoprtion coeffcient
% % is constant and use it to estimate the elastic constants and dielectric
% % permittivity of the liquid crystal.
% 
% a = alpha(-1);
% T_av = ((L_G*a^2 - kappa)*sinh(a) + a*kappa*cosh(a))/(kappa*a^2);
% display(sprintf('Average temperature: %f',T_av));

% Number of voltage values and z coordinate points
nV = length(V);
nz = length(z);
% Define the output variable
theta = zeros(nz,nV);

% Compute the initial condition for the director field as a linear
% interpolation between the boundary conditions with possibly added a
% sinusoidal perturbation of amplitude ampl.
ampl = 1e-2;
theta_init = InitCond(theta_L,theta_R,z,ampl);

% Loop over the voltage values: use the equilibrium configuration for the
% previous voltage as initial guess for the following one.

for k = 1:nV
  % Find the stationary director profile.
  theta(:,k) = FindStationary(@Theta,theta_init,[0.0001,20,4,sqrt(2)],...
                              delta_1,delta_2,alpha_psi,V(k));
  theta_init = theta(:,k);
end